<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReportsController extends Controller
{
    public function index(Request $request)
    {
        $date = $request->query('date', now()->format('Y-m-d'));
        $gradeId = $request->query('grade_id');

        // Get all students (optionally filtered by grade)
        $studentsQuery = DB::table('students')
            ->leftJoin('grades', 'students.grade_id', '=', 'grades.id')
            ->select('students.id', 'students.first_name', 'students.last_name', 'grades.name as grade_name')
            ->where('students.status', 'active');

        if ($gradeId) {
            $studentsQuery->where('students.grade_id', (int)$gradeId);
        }

        $students = $studentsQuery->orderBy('students.last_name')->get();

        // Get attendance for the selected date
        $attendanceData = [];
        foreach ($students as $student) {
            $logs = DB::table('attendance_logs')
                ->where('student_id', $student->id)
                ->whereDate('occurred_at', $date)
                ->orderBy('occurred_at')
                ->get();

            $clockIn = $logs->where('type', 'in')->first();
            $clockOut = $logs->where('type', 'out')->first();

            $attendanceData[] = [
                'student' => $student,
                'clock_in' => $clockIn,
                'clock_out' => $clockOut,
                'status' => $this->getAttendanceStatus($clockIn, $clockOut),
            ];
        }

        $grades = DB::table('grades')->orderBy('display_order')->get();

        return view('reports.index', [
            'attendanceData' => $attendanceData,
            'date' => $date,
            'grades' => $grades,
            'selectedGradeId' => $gradeId,
        ]);
    }

    public function exportCsv(Request $request)
    {
        $date = $request->query('date', now()->format('Y-m-d'));
        $gradeId = $request->query('grade_id');

        $filename = 'attendance_' . $date . ($gradeId ? '_grade_' . $gradeId : '') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function () use ($date, $gradeId) {
            $out = fopen('php://output', 'w');
            fputcsv($out, ['student_name', 'grade', 'clock_in_time', 'clock_out_time', 'status']);

            // Get students
            $studentsQuery = DB::table('students')
                ->leftJoin('grades', 'students.grade_id', '=', 'grades.id')
                ->select('students.id', 'students.first_name', 'students.last_name', 'grades.name as grade_name')
                ->where('students.status', 'active');

            if ($gradeId) {
                $studentsQuery->where('students.grade_id', (int)$gradeId);
            }

            $students = $studentsQuery->orderBy('students.last_name')->get();

            $timeFmt = config('school.time_format', 'H:i');
            foreach ($students as $student) {
                $logs = DB::table('attendance_logs')
                    ->where('student_id', $student->id)
                    ->whereDate('occurred_at', $date)
                    ->orderBy('occurred_at')
                    ->get();

                $clockIn = $logs->where('type', 'in')->first();
                $clockOut = $logs->where('type', 'out')->first();

                fputcsv($out, [
                    $student->last_name . ', ' . $student->first_name,
                    $student->grade_name,
                    $clockIn ? Carbon::parse($clockIn->occurred_at)->format($timeFmt) : '',
                    $clockOut ? Carbon::parse($clockOut->occurred_at)->format($timeFmt) : '',
                    $this->getAttendanceStatus($clockIn, $clockOut),
                ]);
            }

            fclose($out);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function getAttendanceStatus($clockIn, $clockOut): string
    {
        if (!$clockIn) {
            return 'Absent';
        }

        // Read from config (env-driven) instead of DB settings
        $schoolStart = config('school.school_start', '08:00'); // HH:MM 24h
        $schoolEnd = config('school.school_end', '15:00');     // HH:MM 24h
        $graceMinutes = (int) config('school.grace_minutes', 15);

        $clockDate = Carbon::parse($clockIn->occurred_at)->format('Y-m-d');
        $clockInAt = Carbon::parse($clockIn->occurred_at);
        $lateThreshold = Carbon::parse("$clockDate $schoolStart")->addMinutes($graceMinutes);

        $status = 'Present';
        if ($clockInAt->greaterThan($lateThreshold)) {
            $status = 'Late';
        }

        if ($clockOut) {
            $clockOutAt = Carbon::parse($clockOut->occurred_at);
            $endAt = Carbon::parse("$clockDate $schoolEnd");
            if ($clockOutAt->lessThan($endAt)) {
                $status .= ' (Early departure)';
            }
        }

        return $status;
    }

    // Legacy: settings now driven by env/config('school.*'). Keeping method in case of future use.
    private function getSettings(): array
    {
        return [];
    }
}
