<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Services\NotificationService;

class SettingsWebController extends Controller
{
    public function index()
    {
        $rows = DB::table('settings')->select('key','value')->get();
        $settings = [];
        foreach ($rows as $r) { $settings[$r->key] = $r->value; }

        // Provide sensible defaults if missing
        $settings = array_merge([
            'school_name' => 'Chalcedony Prime School',
            'app_name' => 'CPS Attendance',
            'timezone' => 'Africa/Lagos',
            'date_format' => 'Y-m-d',
            'time_format' => 'H:i',
            'school_start_time' => '08:00',
            'school_end_time' => '15:00',
            'grace_minutes' => '15',
        ], $settings);

        return view('settings.index', [ 'settings' => $settings ]);
    }

    public function update(Request $request)
    {
        $data = $request->validate([
            'school_name' => 'required|string|max:200',
            'app_name' => 'required|string|max:200',
            'timezone' => 'required|string|max:100',
            'date_format' => 'required|string|max:50',
            'time_format' => 'required|string|max:50',
            'school_start_time' => 'required|string|max:10',
            'school_end_time' => 'required|string|max:10',
            'grace_minutes' => 'required|integer|min:0|max:120',
            'logo_file' => 'nullable|image|mimes:png,jpg,jpeg,webp,svg|max:2048',
        ]);

        foreach ($data as $key => $value) {
            if ($key === 'logo_file') { continue; }
            DB::table('settings')->updateOrInsert(
                ['key' => $key],
                ['value' => (string)$value, 'updated_at' => now(), 'created_at' => now()]
            );
        }

        // Handle logo upload if present (store via public disk: storage/app/public/brand/logo.png)
        if ($request->hasFile('logo_file')) {
            $file = $request->file('logo_file');
            // Ensure public storage symlink exists: php artisan storage:link
            \Illuminate\Support\Facades\Storage::disk('public')->putFileAs('brand', $file, 'logo.png');
        }

        return redirect()->route('settings.index')->with('ok', 'Settings updated');
    }

    public function testWhatsapp(Request $request)
    {
        $data = $request->validate([
            'to_number' => 'required|string|min:8|max:20',
            'message' => 'required|string|max:1000',
        ]);

        $service = new NotificationService();
        $ok = $service->sendTestMessage($data['to_number'], $data['message']);

        if ($ok) {
            return redirect()->route('settings.index')->with('ok', 'Test message queued' . (env('TWILIO_SID') ? ' and sent.' : ' (no credentials, not sent).'));
        }
        return redirect()->route('settings.index')->withErrors(['to_number' => 'Failed to send test message. Check logs/credentials.']);
    }
}
