<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use App\Services\NotificationService;

class AttendanceController extends Controller
{
    /**
     * POST /attendance/clock-in
     */
    public function clockIn(Request $request)
    {
        return $this->logAttendance($request, 'in');
    }

    /**
     * POST /attendance/clock-out
     */
    public function clockOut(Request $request)
    {
        return $this->logAttendance($request, 'out');
    }

    /**
     * POST /attendance/sync-batch
     * Body (from staff-auth mobile app): { items: [ { qr_uid, type: 'in'|'out', occurred_at, local_id? } ] }
     * Note: Device tokens are suspended. We attribute logs to a special 'Staff API' device.
     */
    public function syncBatch(Request $request)
    {
        $data = $request->validate([
            'items' => 'required|array|min:1',
            'items.*.qr_uid' => 'required|string',
            'items.*.type' => ['required', Rule::in(['in', 'out'])],
            'items.*.occurred_at' => 'nullable|date',
            'items.*.local_id' => 'nullable|string',
        ]);

        $deviceId = $this->ensureStaffApiDevice();

        $results = [];
        foreach ($data['items'] as $e) {
            try {
                $logId = $this->writeAttendance($e['qr_uid'], $e['type'], $deviceId, $e['occurred_at'] ?? now());
                $results[] = [
                    'local_id' => $e['local_id'] ?? null,
                    'ok' => true,
                    'log_id' => $logId,
                ];
            } catch (\Throwable $ex) {
                $results[] = [
                    'local_id' => $e['local_id'] ?? null,
                    'ok' => false,
                    'error' => $ex->getMessage(),
                ];
            }
        }

        return response()->json(['ok' => true, 'synced' => collect($results)->where('ok', true)->count(), 'results' => $results]);
    }

    private function logAttendance(Request $request, string $type)
    {
        $payload = $request->validate([
            'qr_uid' => 'required|string',
            'occurred_at' => 'nullable|date',
            'device_id' => 'required|integer|exists:devices,id',
        ]);

        $logId = $this->writeAttendance(
            $payload['qr_uid'],
            $type,
            $payload['device_id'],
            $payload['occurred_at'] ?? now()
        );

        return response()->json(['log_id' => $logId, 'status' => 'recorded']);
    }

    private function writeAttendance(string $qrUid, string $type, int $deviceId, $occurredAt)
    {
        $student = DB::table('students')->where('qr_uid', $qrUid)->first(['id']);
        if (!$student) {
            abort(404, 'Student not found');
        }

        $logId = DB::table('attendance_logs')->insertGetId([
            'student_id' => $student->id,
            'device_id' => $deviceId,
            'type' => $type,
            'occurred_at' => is_string($occurredAt) ? $occurredAt : $occurredAt,
            'received_at' => now(),
            'status' => 'sent',
            'meta' => null,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // Send WhatsApp notification
        try {
            $notificationService = new NotificationService();
            $templateType = $type === 'in' ? 'clock_in' : 'clock_out';
            $notificationService->sendAttendanceNotification($student->id, $templateType);
        } catch (\Exception $e) {
            // Log but don't fail the attendance recording
            \Log::error('Failed to send attendance notification', [
                'student_id' => $student->id,
                'type' => $type,
                'error' => $e->getMessage()
            ]);
        }

        return $logId;
    }

    private function ensureStaffApiDevice(): int
    {
        $existing = DB::table('devices')->where('name', 'Staff API')->first(['id']);
        if ($existing) {
            return (int)$existing->id;
        }
        return (int) DB::table('devices')->insertGetId([
            'name' => 'Staff API',
            'api_token' => hash('sha256', bin2hex(random_bytes(16))),
            'last_seen_at' => now(),
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }
}
