<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $today = now()->toDateString();

        $totalStudents = DB::table('students')->count();
        $activeStudents = DB::table('students')->where('status', 'active')->count();
        $presentToday = DB::table('attendance_logs')
            ->whereDate('occurred_at', $today)
            ->where('type', 'in')
            ->distinct('student_id')
            ->count('student_id');
        $lateToday = $this->countLateToday($today);
        $messagesToday = DB::table('messages')->whereDate('created_at', $today)->count();

        $latest = DB::table('attendance_logs')
            ->join('students', 'attendance_logs.student_id', '=', 'students.id')
            ->leftJoin('grades', 'students.grade_id', '=', 'grades.id')
            ->select(
                'attendance_logs.type',
                'attendance_logs.occurred_at',
                'students.first_name', 'students.last_name',
                'grades.name as grade_name'
            )
            ->orderByDesc('attendance_logs.occurred_at')
            ->limit(12)
            ->get();

        return view('dashboard', [
            'metrics' => [
                'total_students' => $totalStudents,
                'active_students' => $activeStudents,
                'present_today' => $presentToday,
                'late_today' => $lateToday,
                'messages_today' => $messagesToday,
            ],
            'latest' => $latest,
        ]);
    }

    private function countLateToday(string $date): int
    {
        $settings = $this->getSettings();
        $schoolStart = $settings['school_start_time'] ?? '08:00';
        $graceMinutes = (int)($settings['grace_minutes'] ?? 15);
        $lateThreshold = date('H:i:s', strtotime($schoolStart . ' +' . $graceMinutes . ' minutes'));

        $logs = DB::table('attendance_logs')
            ->select('student_id', DB::raw('MIN(occurred_at) as first_in'))
            ->whereDate('occurred_at', $date)
            ->where('type', 'in')
            ->groupBy('student_id')
            ->get();

        $late = 0;
        foreach ($logs as $log) {
            $t = date('H:i:s', strtotime($log->first_in));
            if ($t > $lateThreshold) $late++;
        }
        return $late;
    }

    private function getSettings(): array
    {
        $rows = DB::table('settings')->select('key', 'value')->get();
        $settings = [];
        foreach ($rows as $row) { $settings[$row->key] = $row->value; }
        return $settings;
    }
}
